import { NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function PATCH(
  request: Request,
  { params }: { params: { id: string } }
) {
  try {
    const body = await request.json()
    const { status, adminNotes } = body

    if (!status) {
      return NextResponse.json(
        { error: 'Status is required' },
        { status: 400 }
      )
    }

    const updateData: any = {
      status,
      updatedAt: new Date(),
    }

    if (adminNotes !== undefined) {
      updateData.adminNotes = adminNotes
    }

    if (status === 'CONFIRMED') {
      updateData.confirmedAt = new Date()
    }

    const order = await db.order.update({
      where: {
        id: params.id,
      },
      data: updateData,
      include: {
        game: true,
        pricingPlan: true,
      },
    })

    return NextResponse.json(order)
  } catch (error) {
    console.error('Failed to update order:', error)
    return NextResponse.json(
      { error: 'Failed to update order' },
      { status: 500 }
    )
  }
}

export async function GET(
  request: Request,
  { params }: { params: { id: string } }
) {
  try {
    const order = await db.order.findUnique({
      where: {
        id: params.id,
      },
      include: {
        game: true,
        pricingPlan: true,
      },
    })

    if (!order) {
      return NextResponse.json(
        { error: 'Order not found' },
        { status: 404 }
      )
    }

    return NextResponse.json(order)
  } catch (error) {
    console.error('Failed to fetch order:', error)
    return NextResponse.json(
      { error: 'Failed to fetch order' },
      { status: 500 }
    )
  }
}